"""MyCmd, cmd wrapper with more features for console-style applications.
"""
"""
Copyright (C) 2011-2024 Doug Lee

This program is free software: you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation, either version 3 of the License, or (at your
option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with this program.  If not, see <http://www.gnu.org/licenses/>.

"""

# Note: We need to unset the TZ environment variable on Windows which is set
# by Cygwin using the UNIX naming convention (e.g. "Australia/Sydney") as
# this will be interpreted by the underlying Windows system using a
# completely different specification language (e.g. "EST+10:00EDT") thus
# creating timezone havoc. Leaving it empty results in the local timezone
# being used which is what we want. Also note that we need to do this before
# the time or datetime modules have been imported or it has no effect.
# The below snippet is copied from a message archive: http://code.activestate.com/lists/python-win32/12335/
import sys, os
if sys.platform == 'win32':
	_tz = os.getenv('TZ')
	if _tz is not None and '/' in _tz:
		del os.environ['TZ']

# Full locale handling using user defaults.
import locale
try: locale.setlocale(locale.LC_ALL, '')
except Exception as e: print(f"Warning, locale setup failed: {str(e)}")

import subprocess, time, re, shlex, tempfile
from cmd import Cmd
import argparse
import threading
import urllib.request
try: from win32api import SetConsoleTitle
except ImportError: pass
import __main__
# Most callers should use the conf from this library...
try: from .conf import conf
# But those who don't may define their own.
except ModuleNotFoundError: from conf import conf

# For Say on Windows.
Dispatch = None
SayTools = None
try:
	import pythoncom
	from win32com.client import Dispatch
	SayTools = Dispatch("Say.Tools")
except Exception: pass

# shlex monkey patch before Python 3.8
try: shlex.join
except AttributeError:
	def __join(lst): return " ".join([shlex.quote(e) for e in lst])
	shlex.join = __join

# Thread-safe printing, for all that go through the next function.
printLock = threading.Lock()
def safePrint(*args, **kwargs):
	with printLock: print(*args, **kwargs)

class classproperty:
	"""Allows for a class-level property. Example: speakEvents.
	Reference: answer 1 in http://stackoverflow.com/questions/3203286/how-to-create-a-read-only-class-property-in-python, 2017-04-04.
	"""
	def __init__(self, getter):
		self.getter = getter
	def __get__(self, instance, owner):
		return self.getter(owner)

class CommandError(Exception):
	"""An error generated by command-handling code. Output is cleaner for non-Python-savvy users.
	"""
	pass
	#def __repr__(self): return __str__(self)

class ArgumentParser(argparse.ArgumentParser):
	"""An override that blocks program exit on error.
	"""
	def exit(self, status=0, msg=""):
		raise CommandError(msg)

# Globals used by the MyCmd class.
justOneCommand = False
screen_lastSec = 0

class MyCmd(Cmd):
	"""Custom wrapper for the cmd.Cmd class.
	Includes window title setting under Windows when win32 is available.
	Include a doc string in the main module; it is used as intro and version command text.
	Initialize as for cmd.Cmd(),
	define conf.name and conf.version (requires a conf module),
	optionally call .allowPython(True) to allow bang (!) Python escapes,
	then run with .run().
	Make do_*(self, line) methods to create commands.
	Override emptyline() if it should do more than just print a new prompt.
	The following commands are defined already:
		help, ?: Print list of commands.
		help or ? commandName: Provide help for specific commands from their do_*() method doc strings.
		about: Print the same intro as printed on startup.
		EOF, quit, exit:  Exit the interpreter. Run() returns.
		clear, cls:  Clear the screen if possible.
		errTrace: Print a traceback of the latest error encountered.
	The following static and class methods are defined for easing command implementation:
		getargs: Parse line into args.
		msg: Print all passed arguments through the right output channel.
		msgNoTime: Like msg but avoids printing timestamps if otherwise printing.
		msgFromEvent: Msg version intended for asynchronous event output.
		dequote: Remove starting and ending quotes on a line if present (rarely needed).
		msgError: Msg only intended for error message output.
		confirm: Present a Yes/No confirmation prompt.
		input_withoutHistory: input that tries not to include its input in readline history.
		getMultilineValue:  Get a multiline value ending with "." on its own line.
		linearList:  Print a list value nicely and with a name.
			Items are sorted (case not significant) and wrapped as necessary.
			Null elements are ignored.
			A filter can be passed to avoid nulls and/or rewrite items.
		selectMatch:  Let the user pick an item from a passed list.
			A custom prompt can also be passed,
			as can a translator function for adjusting how list items sort/print.
		callWithRetry:  Wrapper for subprocess launch functions that retries as needed on Cygwin.
	There are also these object methods:
		dispatchSubcommand(prefix, args): Dispatch a subcommand in args[0] by calling "prefix%s" % (args[0]).
			Example: self.dispatchSubcommand("account_", ["list", "all"]) calls self.account_list(["all"])
	"""

	# .ini section name where user-defined aliases are housed.
	AliasSect = "Aliases"

	def allowPython(self, allow=True):
		"""Decide whether or not to allow bang (!) Python escapes, for
		evaluating expressions and executing statements from the command line.
		"""
		if allow:
			self.do_python = self._do_python
		else:
			try: del self.do_python
			except Exception: pass

	def _fixLine(self, line, doHelp):
		"""Helper for onecmd that handles both commands and help requests.
		"""
		# Handle aliases and other user-defined command-line substitutions.
		if not doHelp: line = self._doSubs(line, doHelp)
		if not line:
			return line
		if line[0] == "!" and "do_python" in self.__dict__:
			line = "python " +line[1:]
		cmd,args,line = self.parseline(line)
		if not cmd: return line
		try:
			cmd1 = self._commandMatch(cmd)
		except (CommandError, KeyError, ValueError):
			# No matches found.
			pass
		else:
			line = line.replace(cmd, cmd1, 1)
			cmd = cmd1
		if not doHelp and cmd.lower() in ["?", "help"]:
			line = line.replace(cmd, "", 1).lstrip()
			line = "help " +self._fixLine(line, True)
		return line

	def _doSubs(self, cmd, doHelp):
		"""Handle aliases and any other user-defined command-line substitutions.
		"""
		if not cmd.strip(): return cmd
		cmd = self._translateAliases(cmd, doHelp)
		return cmd

	def _translateAliases(self, cmd, getHelp):
		"""Translate the first word of cmd if it's an alias, and return the result. %1-style parameters are supported.
		If cmd does not start with an alias, it is returned unchanged.
		Aliases are allowed to refer to other aliases, up to 16 levels.
		"""
		try: aliases = conf[self.AliasSect]
		except Exception: return cmd
		for level in range(1, 17):
			try: cmdword,cmdrest = cmd.split(None, 1)
			except ValueError: cmdword,cmdrest = (cmd,"")
			#safePrint("\n".join(["{0} aliased to {1}".format(alias[0], alias[1]) for alias in aliases.items()]))
			try:
				cmdword = [alias[1] for alias in aliases.items() if alias[0].lower() == cmdword.lower()][0]
				# That throws an IndexError if cmdword is not aliased,
				# so now cmdword is the whole alias RHS or remains what it was before.
				if getHelp: return "Aliased to " +cmdword
				if re.search(r'%\d', cmdword):
					# Positional parameters exist.
					args = shlex.split(cmdrest)
					f = lambda m: self._getArg(m, args)
					cmd = re.sub(r'%(\d+)-?', f, cmdword)
				else: cmd = " ".join([cmdword, cmdrest]).strip()
			except (KeyError, IndexError):
				break
		if level >= 16:
			raise SyntaxError("Too many levels of alias expansion")
		return cmd

	def _getArg(self, m, args):
		"""Positional parameter handler: Returns the arg(s) indicated by m, which is a match object.
		Helper for _translateAliases().
		"""
		# -1 makes alias parameters like %1 1-based instead of 0-based.
		argno = int(m.groups()[0]) -1
		if m.group().endswith("-"):
			# %n- just returns "" if there aren't n+1 args.
			return shlex.join(args[argno:])
		# but %n requires that at least n args are present.
		return args[argno]

	def do_print(self, line=""):
		"""Print a line of text.
		"""
		self.msg(line)

	def do_alias(self, line=""):
		"""Defines or shows one or all aliases or removes one or more aliases.  Examples:
			alias cl clear
			cl  (clears the screen)
			alias cl  (shows that alias)
			alias  (lists all aliases)
			alias -cl  (removes the indicated aliases)
		Parameters are also allowed:
			alias pr1 print Parameter 1 is %1
			pr1 blah  (same as print Parameter 1 is blah)
			alias pr1n print Parameter 1 is %1 and the rest are %2-
			pr1n blah a b c  (same as print Parameter 1 is blah and the rest are a b c)
		"""
		if not line:
			self.msg(self._aliases())
			return
		removing = False
		if line[0] == "-":
			line = line[1:]
			if not line:
				raise SyntaxError("Must specify an alias to remove")
			removing = True
		args = shlex.split(line)
		# These or any prefix of any of these may not be aliased, to avoid user accidents.
		# If a user really wants to alias one of these, it must be done via a direct .ini file edit.
		# Removal is allowed to remove such aliases in case they are so defined.
		forbidden = ["about", "alias", "errtrace", "eof", "exit", "help", "quit"]
		if not removing and args and any([al for al in forbidden if al.startswith(args[0].lower())]):
			raise ValueError(args[0] +" is not allowed as an alias")
		try: aliases = conf[self.AliasSect]
		except Exception: aliases = []
		if removing:
			# Allow multiple aliases to be removed at once.
			# Syntactic leniency: When removing aliases, the leading dash on non-first aliases is optional.
			# Minor side effect: "alias --blah" becomes the same as "alias -blah" here.
			args = [re.sub(r'^-', '', arg) for arg in args]
			# In case of manual file modification.
			conf.load()
			for arg in args:
				al = arg.lower()
				try: del conf.user_cfg[self.AliasSect][al]
				except KeyError: self.msg("No {0} alias".format(al))
				else: self.msg("Alias {0} removed".format(al))
			conf.user_cfg.write()
			conf.load()
			return
		elif len(args) == 1:
			# Just asking for the value of an alias.
			try: val = aliases[args[0].lower()]
			except KeyError:
				self.msg("No {0} alias".format(args[0].lower()))
				return
			self.msg("{0} aliased to {1}".format(args[0].lower(), val))
			return
		# Only adding a new alias remains.
		conf.load()
		lhs = args[0].lower()
		rhs = line.split(None, 1)[1]
		ucfg = conf.user_cfg
		ucfg.setdefault(self.AliasSect, {})
		ucfg[self.AliasSect][lhs] = rhs
		self.msg("{0} aliased to {1}".format(lhs, rhs))
		ucfg.write()
		conf.load()

	def _aliases(self):
		"""Lists all defined aliases and their values.
		"""
		try: aliases = conf[self.AliasSect]
		except Exception: return "No aliases"
		s = "Aliases:"
		found = False
		for al in sorted(aliases.keys()):
			found = True
			val = aliases[al]
			s += "\n   %s = %s" % (al, val)
		if not found: return "No aliases"
		return s

	def onecmd(self, line):
		"""Wrapper for Cmd.onecmd() that handles aliases and errors.
		"""
		try:
			line = self._fixLine(line, False)
			result = Cmd.onecmd(self, line)
			return result
		except KeyboardInterrupt:
			self.msg("Keyboard interrupt")
		except Exception as e:
			self.msg(err())
			return

	@classmethod
	def lineFromArgs(cls, args):
		"""Build a line from args such that args are properly quoted for parsing back into a list.
		Used to execute a command from the command line of this app.
		"""
		args1 = []
		for arg in args:
			if True: arg = cls._fixParm(arg)
			elif len(arg) == 0 or " " in arg or "\t" in arg or "\r" in arg or "\n" in arg:
				if '"' not in arg: arg = '"' +arg +'"'
				elif "'" not in arg: arg = "'" +arg +"'"
				else: arg = cls._fixParm(arg)
			args1.append(arg)
		line = " ".join(args1)
		return line

	@classmethod
	def _fixParm(cls, parm):
		"""Quote parm if necessary so it can be added to a command line and parsed back into an arg later.
		"""
		if parm is None or parm == "": return '""'
		parm = parm.replace('"', r'\"').replace(r"\'", "'")
		if len(parm) == 0 or " " in parm or "\t" in parm or "\r" in parm or "\n" in parm:
			parm = '"' +parm +'"'
		return parm

	def dispatchSubcommand(self, prefix, args):
		"""Dispatch a subcommand in args[0] by calling "prefix%s" % (args[0]).
		Example: self.dispatchSubcommand("account_", ["list", "all"]) calls self.account_list(["all"])
		args may also be a string, in which case the subcommand should be the first argument according to shlex.split().
		When this is the case, args as passed to the subcommand function will also be a string.
		"""
		try:
			if isinstance(args, str):
				a1 = shlex.split(args)
				cmd = a1.pop(0)
				args = shlex.join(a1)
			else:
				cmd = args.pop(0)
			cmd = self._commandMatch(cmd, prefix)
		except (IndexError, KeyError, ValueError):
			cmds = self._commands(prefix)
			cmds = [cmds[cmd] for cmd in sorted(cmds.keys())]
			raise CommandError("Subcommand must be one of: {0}".format(", ".join(cmds)))
		func = eval("self.{0}{1}".format(prefix, cmd))
		return func(args)

	def do_quit(self, line):
		"""Exit the program.  EOF, quit, and exit are identical.
		"""
		self.msg("Quit")
		return True

	def do_eof(self, line):
		"""Exit the program.  EOF, quit, and exit are identical.
		"""
		return self.do_quit(line)

	def do_exit(self, line):
		"""Exit the program.  EOF, quit, and exit are identical.
		"""
		return self.do_quit(line)

	def _do_python(self, line):
		"""Evaluate a Python expression or statement.
		Usage: python expr or python statement.
		Shorthand: !expr or !statement.
		Examples:
			!2+4
			!d = dict()
		Statements and expressions are evaluated in the context of __main__.
		"""
		result = None
		import __main__
		try: result = eval(line, __main__.__dict__)
		except SyntaxError:
			exec(line, __main__.__dict__)
			result = None
		self.msg(str(result))

	def do_errTrace(self, e=None):
		"""Provides a very brief traceback for the last-generated error.
		The traceback only shows file names (no paths) and line numbers.
		"""
		if not e: e = None
		self.msg(errTrace(e))

	def emptyline(self):
		"""What happens when the user presses Enter on an empty line.
		This overrides the default behavior of repeating the last-typed command.
		"""
		return False

	def do_help(self, line):
		"""Print help for the program or its commands/subcommands.
		"""
		if not line.strip():
			# Top-level help (command list etc.).
			return Cmd.do_help(self, line)
		# Handle aliases.
		line1 = self._doSubs(line, True)
		if line1 != line:
			# Print the alias expansion as its help.
			self.msg(line1)
			return False
		line1 = line
		cmd = ""
		level = 1
		while line1:
			parts = line1.split(None, 1)
			word = parts.pop(0)
			line1 = parts[0] if parts else ""
			if level == 1:
				# Normal cmd module situation: A top-level command with a method.
				cmd = f"do_{word}"
			elif level == 2:
				# First sublevel; do_ must be removed.
				cmd = f"{cmd[3:]}_{word}"
			else:
				# Second and subsequent levels; do_ already removed.
				cmd = f"{cmd}_{word}"
			matches = [a for a in dir(self) if a.lower().startswith(cmd.lower())]
			if len(matches) != 1:
				return Cmd.do_help(self, line)
			# Natively-cased full name for this level's method.
			cmd = matches[0]
			level += 1
		try: txt = getattr(self, cmd).__doc__
		# That fails if (1) the method/submethod doesn't exist, or (2) it has no docstring.
		except AttributeError:
			# Let cmd handle this natively.
			return Cmd.do_help(self, line)
		# Print the formatted docstring for the found method.
		self.msg(self._formatHelp(txt))

	def _formatHelp(self, txt):
		"""Format help text for output.
		Tabs are translated to three spaces each.
		CR/LF pairs become just Newline.
		An attempt is made to remove indentation caused by Python class structure.
		If there is an initial blank line, it is removed.
		"""
		txt = txt.replace('\13\10', '\n').replace('\t', '   ')
		# Assume doc block starts immediately, not after a Newline.
		firstIndented = False
		if txt.startswith('\n'):
			# But it doesn't.
			txt = txt[1:]
			firstIndented = True
		# Find the smallest indent so we can remove it from all lines.
		# If the first line started without a leading Newline, ignore that one.
		ignoreNext = not firstIndented
		noMinIndent = 99999
		minIndent = noMinIndent
		for line in txt.splitlines():
			if ignoreNext:
				ignoreNext = False
				continue
			lineIndent = len(line) -len(line.lstrip())
			minIndent = min(minIndent, lineIndent)
		# Now apply what we found.
		if minIndent != noMinIndent:
			lines = []
			ignoreNext = not firstIndented
			for line in txt.splitlines():
				if ignoreNext:
					lines.append(line)
					ignoreNext = False
					continue
				line = line[minIndent:]
				lines.append(line)
			txt = "\n".join(lines)
		return format(txt)

	def run(self):
		"""Kick off the command-line interpreter.
		The command line of the program is allowed to consist of one command,
		in which case it is run and the program exits with its return code,
		or with 1 if the command returns a non-empty, non-integer value.
		Otherwise, the normal command loop is started.
		This call returns on quit/exit/EOF,
		or in the one-command-on-app-command-line case, after the command runs.
		The prompt is conf.name unless the prompt() property is overridden in a subclass.
		name is the name for the intro ("type 'help' for help") line.
		sys.argv is processed by this call,
		so if you have something to do with it, do it before calling.
		"""
		global justOneCommand, screen_lastSec
		try: __main__.__doc__
		except Exception: sys.exit("The main module must define a doc string to be used as the startup screen")
		try: conf.name, conf.version
		except Exception: sys.exit("The main module must set conf.name and conf.version")
		self.plat = self._getPlatform()
		args = sys.argv[1:]
		justOneCommand = False
		screen_lastSec = 0
		if args:
			# Do one command (without intro or prompt) and exit.
			# The command's return value is returned to the OS.
			justOneCommand = True
			line = self.lineFromArgs(args)
			sys.exit(self.onecmd(line))
		vr = "version" if "." in conf.version else "revision"
		name = '{}\n\n{} {} {}, type "help" or "?" for help.'.format(
			__main__.__doc__.strip(),
			conf.name,
			vr,
			conf.version
		)
		try: SetConsoleTitle(conf.name)
		except Exception: pass
		self.cmdloop(name)

	@property
	def prompt(self):
		p = conf.name
		c = pendingMessageCount()
		if c:
			p = f"{p} ({c})"
		p = f"{p}> "
		return p

	@staticmethod
	def _getPlatform():
		"""Get a string representing the type of OS platform we're running on.
	"""
		plat = sys.platform.lower()
		if plat[:3] in ["mac", "dar"]:
			return "mac"
		elif "linux" in plat:  # e.g., linux2
			return "linux"
		elif "win" in plat:  # windows, win32, cygwin
			return "windows"
		return plat

	@staticmethod
	def getPlatInfo():
		"""Return some info about the current platform as a string.
		Format for Windows: platform, winverstring maj.min.bld extra
			where "extra" is service packs and such.
			Example: win32, WinXP 5.1.2600 Service Pack 3
		Format for Mac and Linux: platform (may be more to come later).
		"""
		plat = sys.platform
		try:
			maj,min,bld,pl,extra = sys.getwindowsversion()
			try: pl = ["Win32s/Win3.1", "Win9x/Me", "WinNT/2000/XP/x64", "WinCE"][pl]
			except IndexError: pl = "platform " +str(pl)
			pl += " %d.%d.%d %s" % (maj, min, bld, extra)
			plat += ", " +pl
		except AttributeError: pass
		return plat

	@classmethod
	def launchURL(cls, url):
		"""Launch the given URL in a browser.  May not be supported on all platforms."
		Provides some protection against calls with non-URL strings.
		Use a browser= setting in Settings in the config file to select a specific browser.
		Raises a CalledProcessError with a returncode attribute if unsuccessful.
		Raises a RuntimeError if not supported on this platform.
		Returns True on success for compatibility with older code that expected a True/False result instead of exceptions.
		"""
		try: urltype = url.split(":", 1)[0].lower()
		except Exception: urltype = ""
		if len(urltype) < 1 or not re.match(r'^[a-z0-9_]+$', urltype):
			raise ValueError("URL type not recognized")
		url = url.replace(" ", "%20").replace('"', "%22").replace("'", "%27")
		try: browser = conf["Settings"]["browser"]
		except Exception: browser = None
		plat = sys.platform
		if browser:
			cmds = ([browser, url],)
		elif plat == "cygwin" or plat.startswith("win"):
			# On Cygwin, `cygstart' works but prevents Windows auto-login from working.
			# Running Explorer directly fixes that, at least on Windows XP.
			# [DGL, 2009-03-17]
			# It also works on ActivePython.
			cmds = (["explorer", url],)
		elif "linux" in plat.lower():
			# wslview enables use of default Windows browser on WSL.
			cmds = (["wslview", url], ["xdg-open", url], ["firefox", url], ["lynx", url], ["links", url], ["w3m", url],)
		elif plat == "darwin":  # MacOS
			cmds = (["open", url],)
		else:
			raise RuntimeError("LaunchURL not supported on this platform")
		cls.msg("Web page launching.")
		for i,cmd in enumerate(cmds):
			try:
				subprocess.check_call(cmd)
				# Quit trying alternatives if that did not throw an exception.
				break
			except Exception as e:
				# Windows Explorer returns 1 on success! [DGL, 2017-09-30, Windows 10]
				if (isinstance(e, subprocess.CalledProcessError)
				and (plat == "cygwin" or plat.startswith("win"))
				and e.returncode == 1):
					# Call that a success by exiting the loop.
					break
				elif i+1 == len(cmds):
					# Fail if this is the last command to try.
					raise
				# Otherwise just try the next command quietly.
		# One of the commands succeeded.
		return True

	def rel(self, url1, url2, line="", start=0):
		# The below doc string is user documentation for help!
		# To use, define do_rel() and pass required url1 and url2 for main and beta release docs, respectively.
		# Example: self.rel(main_url, beta_url, line, start=75000)
		# For docs, below do_rel(), add do_rel.__doc__ = MyCmd.rel.__doc__
		# line comes from the line parameter to do_rel().
		# start is an optional number of bytes to skip that safely won't contain the start of Revision History.
		"""Show the release notes for any updates to this application.
		Without arguments, shows release notes for revisions that are pending (not installed yet).
		If -b is specified, checks for beta revisions instead of releases.
		If a negative number like -1 is specified, prints revisions newer than the revision that far before the current one.
		If a positive number is specified, prints revisions newer than that specific revision.
		These last two syntaxes allow checking of what changes were recently installed.
		"""
		# Assumptions that must be true for this to work:
		#	* The passed URLs must be correct for the app's user guide (release and beta).
		#	* The user guide's last h2 heading must be the revision history.
		#	* The revision history must consist of h3 headings for each release.
		#	* Other than tags P, UL, and LI, removal of all tags must not break the content.
		#	* UL tags must not nest more than one level deep, and OL and DL tags must not be used.
		url = url1
		cutrev = int(conf.version)
		args = shlex.split(line)
		for arg in args:
			if arg == "-b":
				url = url2
			elif arg.isdigit():
				cutrev = int(arg)
			elif arg[0] == "-" and arg[1:].isdigit():
				cutrev += int(arg)
		hdrs = None
		if start > 0:
			hdrs = {'bytes': f"{start:d}-"}
		req = urllib.request.Request(url, headers=hdrs)
		with urllib.request.urlopen(req) as stream: txt = stream.read().decode("UTF-8")
		# End of Revision History heading through end of document content kept.
		txt = txt.rsplit("</h2>", 1)[1].rsplit("</body>", 1)[0].strip()
		# Further minus the material between the Revision History heading and the first release entry.
		txt = "<h3" +txt.split("<h3", 1)[1]
		# No newlines from source; we plant those based on tags.
		txt = txt.replace("\r\n", "\n").replace("\n", " ")
		# Convert tags we need to track into internal equivalents.
		txt = re.sub('(?i)<h3.*?>', '\nh3:', txt)
		txt = re.sub('(?i)</h3>', '', txt)
		txt = re.sub('(?i)<p/?>', '\np:', txt)
		txt = re.sub('(?i)</p>', '', txt)
		txt = re.sub('(?i)<(/?ul)>', '\n\\1\n', txt)
		# LI tags start their own lines.
		txt = re.sub('(?i)(<li>)', '\nli:', txt)
		# All other tags are stripped.
		# Warning: This includes links, so the normal link material will appear with no indication that it was originally a link.
		txt = re.sub(r'<\/?[a-zA-Z0-9]+.*?>', '', txt)
		txt = re.sub(r'\n\n+', r'\n', txt)
		outlines = []
		ul = 0
		for line in txt.splitlines():
			line = line.strip()
			if not line: continue
			if line.startswith("h3:"):
				line = "***** " +line[3:]
				rv = line.split(None, 2)[2].split(None, 1)[0].replace(",", "")
				try: rv = int(rv)
				except ValueError: break
				if rv <= cutrev: break
				if len(outlines): outlines.append("")
				outlines.append(line)
			elif line == 'ul':
				ul += 1
			elif line == '/ul':
				ul -= 1
			elif line.startswith("li:"):
				line = line.replace("li:", "* " if ul==1 else "    - ", 1)
				outlines.append(line)
			else:
				outlines.append(line)
		if not len(outlines):
			self.msg("No new revisions available according to the online release notes.")
			return
		self.msg("\n".join(outlines), indent1="", indent2="  ")

	def debugging(self):
		"""Return True if debugging is enabled (via a debug key in Settings in the .ini file).
		"""
		try: return bool(conf["Settings"]["debug"])
		except Exception: return False

	@staticmethod
	def callWithRetry(func, *args, **kwargs):
		"""For Cygwin 1.8 on Windows:
		Forks can ffail randomly in the presence of things like antivirus software,
		because DLLs attaching to the process can cause address mapping problems.
		This function retries such calls so they don't fail.
		"""
		i = 1
		while i <= 50:
			try:
				return func(*args, **kwargs)
			except OSError as e:
				i += 1
				safePrint("Retrying, attempt #" +str(i))
		safePrint("Retry count exceeded.")

	def do_about(self, line):
		"""
		Report the app name and version info (same as what prints on startup).
		"""
		self.msg(self.intro)

	def do_clear(self, line):
		"""
		Clears the screen.
		"""
		if self.plat == "windows" and sys.platform != "cygwin":
			os.system("cls")
			return ""
		os.system("clear")
		return ""

	def do_cls(self, line):
		"""
		Clears the screen.
		"""
		return self.do_clear(line)

	def _commands(self, prefix="do_"):
		"""Returns all available commands or subcommands as indicated by methods in this class with the given prefix.
		The return value is a dict where keys are lower-case commands and values are mixed-case as appearing in code as function names.
		The default prefix is "do_", which returns a list of commands available to the user.
		"""
		cmds = {}
		for funcname in [f for f in dir(self) if f.startswith(prefix)]:
			cmd = funcname[len(prefix):]
			cmds[cmd.lower()] = cmd
		return cmds

	def _commandMatch(self, cmdWord, prefix="do_"):
		"""
		Returns all available commands or subcommands, and the exact command word indicated by cmdWord.
		Implements command matching when an ambiguous command prefix is typed.
		"""
		cmds = self._commands(prefix)
		# An exact match wins even if there are longer possibilities.
		try: return cmds[cmdWord.lower()]
		except KeyError: pass
		# Get a list of matches, capitalized as they are in the code function names.
		cmdWord = cmdWord.lower()
		matches = [f for f in list(cmds.keys()) if f.startswith(cmdWord)]
		matches = [cmds[cmdKey] for cmdKey in matches]
		if len(matches) == 1: return matches[0]
		elif len(matches) == 0: raise CommandError('No valid command matches "{0}"'.format(cmdWord))
		return self.selectMatch(matches, "Which command did you mean?")

	@classmethod
	def confirm(cls, prompt):
		"""Get permission for an action with a y/n prompt.
		Returns True if "y" is typed and False if "n" is typed.
		Repeats request until one or the other is provided.
		KeyboardInterrupt signals equate to "n"
		"""
		if not prompt.endswith(" "): prompt += " "
		l = ""
		while not l:
			l = cls.input_withoutHistory(prompt)
			l = l.strip()
			l = l.lower()
			if l == "keyboardinterrupt": l = "n"
			if l in ["n", "no"]: return False
			elif l in ["y", "yes"]: return True
			cls.msg("Please enter y or n.")
			l = ""

	@classmethod
	def selectMatch(cls, matches, prompt=None, ftran=None, allowMultiple=False, sort=True, promptOnSingle=False):
		"""
		Return one or more matches from a set.
		matches: The set of matches to consider.
		prompt: The prompt to print above the match list. If none is provided, a reasonable default is used.
		ftran: The function on a match to make it into a string to print. If not provided, no translation is performed and each match must natively be printable.
		allowMultiple: If True, lets the user enter multiple numbers and returns a list of matches. Defaults to False.
			When this is set, the user may also type the word "all" to select the entire list.
			An exclamation (!) before a list of numbers means select all but the listed ones.
		sort: True by default to sort entries, False to leave unsorted.
		promptOnSingle: If True, the user is prompted for a selection even when there is only one option. Defaults to False.
		"""
		if not ftran: ftran = lambda m: m
		mlen = len(matches)
		if mlen == 0: raise CommandError("No matches found")
		if mlen == 1 and not promptOnSingle:
			if allowMultiple: return [matches[0]]
			else: return matches[0]
		if sort: matches = sorted(matches, key=ftran)
		try:
			mlist = [str(i+1) +" " +ftran(match) for i,match in enumerate(matches)]
		except (TypeError, UnicodeDecodeError):
			mlist = [str(i+1) +" " +str(ftran(match)) for i,match in enumerate(matches)]
		if not prompt:
			if allowMultiple: prompt = "Select one or more options:"
			else: prompt = "Select an option:"
		m = prompt +"\n   " +"\n   ".join(mlist)
		cls.msg(m)
		if allowMultiple: prompt = """Selections (or Enter to cancel), ! to negate, "all" for all: """
		else: prompt = "Selection (or Enter to cancel): "
		l = ""
		while not l:
			l = cls.input_withoutHistory(prompt)
			l = l.strip()
			if not l: break
			if allowMultiple:
				try: return cls._collectSelections(matches, l)
				except (IndexError, SyntaxError):
					# Error message printed by _collectSelections().
					l = ""
					continue
			try:
				if l and int(l): return matches[int(l)-1]
			except IndexError:
				cls.msg("Invalid index number")
				l = ""
		raise CommandError("No option selected")

	def help_selection(self):
		"""Help for how to select items via selectMatch().
		"""
		self.msg("""
Handling Selection Lists:

When a numbered list of options appears followed by a prompt to make one or more selections, you can type a number to select just that item. When multiple selections are supported, the following also work:
* Type numbers separated by spaces and/or commas to select more than one option.
* Type numbers separated by dashes to select a range; for example, 2-5.
* Merge these as needed to select complex sets of options; e.g., 2, 4, 6-9, 12.
* Type a number, list of numbers, range, or combination, all preceded by an exclamation mark (!) to select all but the given option(s). Examples: !5 for all but option 5, and !3 5 6-9 for all but options 3, 5, 6, 7, 8, and 9.
* Type "all" to select all available options.
""".strip())

	@classmethod
	def _collectSelections(cls, matches, l):
		"""
		Return the matches selected by l. Supported syntax examples (numbers are 1-based):
			9: Just the 9th match.
			2,5 or 2 5 or 2, 5: Matches 2 and 5.
			2-5 or 2..5: Matches 2 3 4 and 5.
			2-5, 9 etc.: Matches 2 through 5 and 9.
			!9: All but the 9th match. Works for all other above ranges as well.
			all: All matches.
		"""
		# First some syntactic simplifications and easy cases.
		l = l.strip()
		if l.lower() == "all":
			return matches
		negating = False
		if l.startswith("!"):
			l = l[1:].lstrip()
			negating = True
		# Comma/space combos become a single comma.
		l = re.sub(r'[ \t,]+', r',', l)
		# .. becomes -
		l = re.sub(r'\.\.', '-', l)
		indices = set()
		# Now make units, each being an index or a range.
		units = l.split(',')
		for unit in units:
			if "-" in unit:
				start,end = unit.split("-")
			else:
				start,end = unit,unit
			start = int(start)
			end = int(end)
			if start < 1 or start > len(matches): 
				m = "%d is not a valid index" % (start)
				raise IndexError(m)
			if end < 1 or end > len(matches): 
				m = "%d is not a valid index" % (end)
				raise IndexError(m)
			indices.update(list(range(start-1, end)))
		if negating: indices = set(range(0, len(matches))) -indices
		return [matches[i] for i in sorted(indices)]

	@classmethod
	def msgNoTime(cls, *args):
		kwargs = {"noTime": True}
		cls.msg(*args, **kwargs)

	@classmethod
	def msgFromEvent(cls, *args):
		kwargs = {"fromEvent": True}
		speakEvents = 0
		try: speakEvents = cls.speakEvents
		except Exception: pass
		if not speakEvents: speakEvents = 0
		if int(speakEvents) != 0:
			mq_vo.extend(args)
		cls.msg(*args, **kwargs)

	@classmethod
	def msg(cls, *args, **kwargs):
		"""
		Arbitor of event output message format:
		Prints times as necessary but not every time.
		"""
		global justOneCommand, screen_lastSec
		# This fails on an early exception.
		try: conf = __main__.conf
		except AttributeError: conf = None
		indent1 = kwargs.get("indent1") or None
		indent2 = kwargs.get("indent2") or None
		sf = ""
		try: sf = conf.option("stampInterval")
		except Exception: pass
		if not sf:
			secFreq = 120
		else:
			secFreq = int(sf)
		tm = time.asctime() +":\n"
		now = time.mktime(time.localtime())
		if not screen_lastSec:
			screen_lastSec = now
		s = ""
		started = False
		for item in args:
			if item is None: continue
			if started: s += " "
			started = True
			s += item if type(item) is str else str(item)
		if not started: return
		s1 = s
		# Time can only print for events, when this is not a one-command run, and when it is not suppressed.
		doTime = (not justOneCommand) and (not kwargs.get("noTime")) and kwargs.get("fromEvent")
		# If secFreq is negative, don't print time either.
		if secFreq < 0:
			doTime = False
		# Or if secFreq seconds haven't passed yet since we printed the last stamp.
		elif now - screen_lastSec < secFreq:
			doTime = False
		if doTime:
			s1 = tm +s
			screen_lastSec = now
		s1 = format(s1, indent1=indent1, indent2=indent2)
		if kwargs.get("fromEvent"):
			# Stamp, if any, is included here so that queued events sufficiently announce their times when pulled.
			mq.append(s1)
		else:
			safePrint(s1)

	@classmethod
	def msgErrOnly(cls, *args, **kwargs):
		"""
		msg() but only for errors.
		"""
		if not args or args[0].startswith("ERROR"):
			cls.msg(*args, **kwargs)
		return

	@classmethod
	def getMultilineValue(cls):
		"""
		Get and return a possibly multiline value.
		The content is prompted for and terminated with a dot on its own line.
		An EOF also ends a value.
		"""
		cls.msg("Enter text, end with a period (.) on a line by itself.")
		content = ""
		while True:
			try:
				line = input("")
			except EOFError:
				line = "."
			line = line.strip()
			if line == ".":
				break
			if content:
				content += "\n"
			content += line
		return content

	@staticmethod
	def linearList(name, l, func=lambda e: str(e)):
		"""
		List l on a (possibly long and wrap-worthy) line.
		The line begins with a header with name and entry count.
		Null elements are removed.  If you don't want this, send in a func that doesn't return null for any entry.
		"""
		l1 = sorted([_f for _f in map(func, l) if _f], key=lambda k: k.lower())
		if len(l) == 0:
			return "%s: 0" % (name)
		return "%s (%0d): %s." % (name, len(l1), ", ".join(l1))

	@staticmethod
	def lineList(name, l, func=lambda e: str(e)):
		"""
		List l one line per entry, indented below a header with name and entry count.
		Null elements are removed.  If you don't want this, send in a func that doesn't return null for any entry.
		"""
		l1 = sorted([_f for _f in map(func, l) if _f], key=lambda k: k.lower())
		if len(l1) == 0:
			return "%s: 0" % (name)
		return "%s (%0d):\n    %s" % (name, len(l1), "\n    ".join(l1))

	@staticmethod
	def getargs(line, count=0):
		"""
		Parse the given line into arguments and return them.
		Args are dequoted unless count is non-zero and less than the number of arguments.
		In that case, all but the last arg are dequoted.
		Parsing rules are those used by shlex in Posix mode.
		"""
		# shlex.split dequotes internally.
		if not count or count >= len(line): return shlex.split(line)
		args = shlex.split(line)
		if len(args) < count: return args
		# Dequoted args up to but not including the last.
		args = args[:count]
		# Collect args without dequoting into the last one, then append it.
		tokenizer = shlex.shlex(line)
		[next(tokenizer) for i in range(0, count)]
		lastArg = " ".join([t for t in tokenizer])
		#safePrint("LastArg: " +lastArg)
		args.append(lastArg)
		return args

	@staticmethod
	def dequote(line):
		"""Remove surrounding quotes (if any) from line.
		Also unescapes the quote removed if found inside the remaining string.
		"""
		if not line: return line
		if (line[0] == line[-1] and line[0] in ["'", '"']
		and len(shlex.split(line)) == 1):
			q = line[0]
			line = line[1:-1]
			line = line.replace('\\'+q, q)
		return line

	@staticmethod
	def input_withoutHistory(prompt=None):
		"""
		input() wrapper that keeps its line out of readline history.
		This is to avoid storing question answers like "1."
		"""
		l = input(prompt)
		if len(l) == 0: return l
		try: readline.remove_history_item(readline.get_current_history_length() -1)
		except (NameError, ValueError): pass
		return l

# Input helpers.

def input(prompt=None):
	try:
		return input0(prompt)
	except KeyboardInterrupt:
		return "KeyboardInterrupt"

__builtins__["input0"] = __builtins__["input"]
__builtins__["input"] = input

# Output helpers.

# Formatter for output.
import textwrap
try: from shutil import get_terminal_size
except Exception: pass
fmt = textwrap.TextWrapper()
def format(text, indent1=None, indent2=None, width=None):
	"""
	Format text for output to screen and/or log file.
	Individual lines are wrapped with indent.
	"""
	if not width:
		try: width = get_terminal_size().columns
		except Exception: width = 79
	if indent1 is None:
		indent1 = ""
		if indent2 is None:
			indent2 = "   "
	elif indent2 is None:
		indent2 = indent1 +"   "
	fmt.width = width
	lines = text.splitlines()
	wlines = []
	for line in lines:
		lineIndent = " " * (len(line) -len(line.lstrip()))
		fmt.initial_indent = indent1
		fmt.subsequent_indent = indent2 +lineIndent
		wlines.append("\n".join(fmt.wrap(line)))
	text = "\n".join(wlines)
	return text

class MessageQueue(list):
	def __init__(self, *args, **kwargs):
		self.holdAsyncOutput = False
		speechQueue = False
		if "speechQueue" in kwargs:
			speechQueue = kwargs["speechQueue"]
			del kwargs["speechQueue"]
		self.speechQueue = speechQueue
		list.__init__(self, *args, **kwargs)
		if speechQueue:
			self.thr = threading.Timer(0, self.watch)
			self.thr.setDaemon(True)
			self.thr.start()

	def output(self, nmsgs=0):
		"""
		Output nmsgs messages.
		If nmsgs is not passed, treat as if it were 0.
		If nmsgs is positive, output that many messages.
		If nmsgs is less than 0, output all pending messages.
		If nmsgs is 0:
			- If self.holdAsyncOutput is True, output nothing now.
			- Else, output as if nmsgs were -1.
		"""
		if nmsgs == 0:
			if self.holdAsyncOutput:
				nmsgs = 0
			else:
				nmsgs = -1
		while len(self) and nmsgs != 0:
			s = self.pop(0)
			safePrint(s)
			if nmsgs > 0:
				nmsgs -= 1

	def watch(self):
		while True:
			if len(self) > 0:
				say(" ")
				while len(self):
					m = self.pop(0)
					say(m)
			time.sleep(2.0)

	def append(self, *args, **kwargs):
		list.append(self, *args, **kwargs)
		try: nmsgs = int(conf.optGet("queueMessages", 0, "Options"))
		except Exception: nmsgs = 0
		self.output(nmsgs)

mq = MessageQueue()
mq_vo = MessageQueue(speechQueue=True)

def pendingMessageCount():
	return len(mq)

def flushMessages(nmsgs):
	mq.output(nmsgs)

def err(origin="", exctype=None, value=None, traceback=None):
	"Nice one-line error messages."
	errtype,errval,errtrace = (exctype, value, traceback)
	exctype,value,traceback = sys.exc_info()
	if not errtype: errtype = exctype
	if not errval: errval = value
	if not errtrace: errtrace = traceback
	# Static error trace preservation for errTrace().
	err.val = errval
	err.trace = errtrace
	buf = ""
	if origin: buf += origin +" "
	name = errtype.__name__
	if name == "CommandError": name = ""
	if name: buf += name +": "
	try: buf += str(errval)
	except UnicodeDecodeError: buf += str(errval)
	for i in range(2, len(errval.args)):
		buf += ", " +str(errval.args[i])
	return buf

def errTrace(e=None):
	"""Provides a very brief traceback for the last-generated error.
	The traceback only shows file names (no paths) and line numbers.
	"""
	if e is None:
		try: e = err.trace
		except AttributeError:
			return "No error has been recorded yet."
	trc = []
	while e:
		try: l = e.tb_lineno
		except AttributeError: l = 0
		l = str(l) if l else "<noLineNumber>"
		try: fname = e.tb_frame.f_code.co_filename
		except AttributeError: fname = "<noFileName>"
		fname = os.path.basename(fname)
		trc.append(f"{fname} {l}")
		try: e = e.tb_next
		except AttributeError: e = None
	return ", ".join(trc)

def say(*args):
	"""
	On MacOS, speak via the default Mac voice.
	On Windows. speak via SayTools if available.
	"""
	try: s = " ".join(args)
	except TypeError: s = str(args)
	s = re.sub(r'[A-Z_]+', cleanForSpeech, s)
	plat = sys.platform
	if (plat == "cygwin" or plat.startswith("win")) and SayTools:
		sys.coinit_flags = 0
		pythoncom.CoInitialize()
		try: SayTools.Say(s)
		except Exception: safePrint(__main__.err())
		pythoncom.CoUninitialize()
	elif plat == "darwin": # MacOS
		cmd = ["say",]
		sprefix = os.environ.get("SAYPREFIX")
		if sprefix: s = sprefix +s
		subprocess.Popen(cmd, stdin=subprocess.PIPE, text=True).communicate(s)
	elif "linux" in plat.lower():
		pass

def cleanForSpeech(m):
	"""
	Make a few adjustments to a string to make it sound better.
	Based on the Mac default voice (Alex).
	This is called by re.sub from say().
	"""
	s = m.group()
	return s

